<?php

namespace App\Services\Logistics\Providers;

use App\Adapters\Shipping\UpsAdapter;
use App\Services\Logistics\LogisticsProviderInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class UpsService implements LogisticsProviderInterface
{
    /**
     * Calculate shipping cost.
     *
     * @param array $packageDetails
     * @return array
     */
    public function calculateShippingCost(array $packageDetails): array
    {
        return [
            'logistics' => 'DHL',
            'cost' => 200.00,
            'currency' => 'USD',
            'estimated_delivery' => '5-7 business days'
        ];
    }

    /**
     * Track shipment.
     *
     * @param string $trackingNumber
     * @return array
     */
    public function trackShipment(string $trackingNumber): array
    {
        return [
            'tracking_number' => $trackingNumber,
            'status' => 'In Transit',
            'estimated_delivery' => '2025-02-20',
        ];
    }

    /**
     * Create shipment.
     *
     * @param array $shipmentDetails
     * @return array
     */
    public function createShipment(array $shipmentDetails): array
    {
        return [
            'shipment_status' => 'Created',
            'shipment_details' => $shipmentDetails,
        ];
    }

    public function availablityCheck(array $options): array
    {
        try {
            $upsRequest = UpsAdapter::buildUpsAddress($options);
            $upsToken = $this->getUpsToken();

            $upsResponse = Http::withToken($upsToken)
                ->post(config('services.ups.base_url') . config('services.ups.address_validation_url'), $upsRequest);

            if ($upsResponse->successful() && !empty($upsResponse['XAVResponse']['Candidate'][0]['AddressKeyFormat'])) {
                $candidate = $upsResponse['XAVResponse']['Candidate'][0]['AddressKeyFormat'] ?? [];

                return [
                    'status' => 'success',
                    'source' => 'UPS',
                    'address' => [
                        'streetLines' => $candidate['AddressLine'] ?? [],
                        'city'        => $candidate['PoliticalDivision2'] ?? '',
                        'state'       => $candidate['PoliticalDivision1'] ?? '',
                        'postalCode'  => $this->buildPostalCode(
                            $candidate['PostcodePrimaryLow'] ?? '',
                            $candidate['PostcodeExtendedLow'] ?? ''
                        ),
                        'countryCode' => $candidate['CountryCode'] ?? '',
                    ],
                    'type' => 'BUSINESS', // UPS doesn't always provide this directly
                ];
            }

            return [
                'status' => 'error',
                'source' => 'UPS',
                'message' => 'Failed to resolve address via UPS',
                'data' => $upsResponse->json() // optional: include raw data
            ];
        } catch (\Exception $e) {
            Log::error('UPS API exception', ['message' => $e->getMessage()]);
            return [
                'status' => 'error',
                'source' => 'UPS',
                'message' => 'UPS API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    private function getUpsToken()
    {
        $client_id = config('services.ups.client_id');
        $client_secret = config('services.ups.client_secret');

        $response = Http::withBasicAuth($client_id, $client_secret)
            ->asForm()
            ->post(config('services.ups.base_url') . config('services.ups.token_url'), [
                'grant_type' => 'client_credentials',
            ]);

        if ($response->failed()) {
            abort(500, 'UPS token generation failed');
        }

        return $response->json()['access_token'];
    }

    private function buildPostalCode($base, $addon)
    {
        return $addon ? "$base-$addon" : $base;
    }
}
