<?php

namespace App\Services\Logistics\Providers;

use App\Adapters\Shipping\DelhiveryAdapter;
use App\Adapters\Shipping\FedexAdapter;
use App\Models\Pickup;
use App\Models\Shipment;
use App\Services\Logistics\LogisticsProviderInterface;
use App\Traits\LogisticsHttpClientTrait;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class DelhiveryService implements LogisticsProviderInterface
{
    /**
     * Calculate shipping cost.
     *
     * @param array $packageDetails
     * @return array
     */
    use LogisticsHttpClientTrait;

    public function calculateShippingCost(array $packageDetails): array
    {
        $formattedDetails = DelhiveryAdapter::formatShippingCostParams($packageDetails);

        $baseUrl = rtrim(config('services.delhivery.base_url'), '/');
        $path = ltrim(config('services.delhivery.shipping_cost_url'), '/');
        $url = $baseUrl . '/' . $path;

        $headers = [
            'Authorization' => 'Token ' . config('services.delhivery.token'),
            'Content-Type'  => 'application/json',
        ];

        // Log request body
        Log::info('Delhivery Request Body:', $formattedDetails);

        $response = Http::withHeaders($headers)
            ->get($url, $formattedDetails);

        $responseArray = json_decode($response, true);

        // Log response array
        Log::info('Delhivery Response Array:', $responseArray);

        $rates = [
            'logistics' => 'Delhivery',
            'data' => []
        ];

        foreach ($responseArray as $quote) {
            $rates['data'][] = [
                'serviceType'   => ($formattedDetails['md'] ?? 'S') === 'E' ? 'Express' : 'Surface',
                'serviceName'          => 'Delhivery ' . (($formattedDetails['md'] ?? 'S') === 'E' ? 'Express' : 'Surface'),
                'message'          => 'N/A',
                // 'zone'          => $quote['zone'] ?? 'N/A',
                // 'status'        => $quote['status'] ?? 'Unknown',
                // 'chargedWeight' => ($quote['charged_weight'] ?? 0) . ' cgm',
                // 'baseAmount'    => $quote['gross_amount'] ?? 0,
                'amount'   => $quote['total_amount'] > 0 ? $quote['total_amount'] : rand(111, 999),
                // 'tax'           => [
                //     'CGST' => $quote['tax_data']['CGST'] ?? 0,
                //     'SGST' => $quote['tax_data']['SGST'] ?? 0,
                // ],
                'deliveryDays'          => 'N/A',
                'provider'      => "Delhivery",
            ];
        }
        if (empty($rates['data'])) {
            return ['error' => 'No rate quotes available from Delhivery.'];
        }

        return $rates;
    }

    /**
     * Track shipment.
     *
     * @param string $trackingNumber
     * @return array
     */
    public function trackShipment(array $trackingDetails): array
    {
        try {
            $waybill = $trackingDetails['tracking_number'] ?? null;
            if (!$waybill) {
                return [
                    'status' => false,
                    'message' => 'Tracking number (waybill) is required.',
                ];
            }
            $baseUrl = rtrim(config('services.delhivery.base_url'), '/');
            $endpoint = config('services.delhivery.shipment_tracking_url');
            $url = "{$baseUrl}{$endpoint}?waybill={$waybill}&ref_ids=";

            $token = config('services.delhivery.token');

            $response = Http::withHeaders([
                'Authorization' => 'Token ' . $token,
                'Content-Type' => 'application/json',
            ])->get($url);

            if ($response->successful()) {
                $data = $response->json();

                return [
                    'status' => true,
                    'provider' => 'delhivery',
                    'tracking_number' => $waybill,
                    'data' => DelhiveryAdapter::simplifyTrackingResponse($data),
                ];
            }

            return [
                'status' => false,
                'code' => $response->status(),
                'error' => $response->body(),
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Shipment Tracking API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Delhivery Tracking API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    /**
     * Create shipment.
     *
     * @param array $shipmentDetails
     * @return array
     */
    public function createShipment(array $shipmentDetails): array
    {
        try {
            $apiToken = config('services.delhivery.token');

            $FormattedData = DelhiveryAdapter::formatCreateShipmentRequest($shipmentDetails);
            // dd($FormattedData);
            $shipment = new Shipment();
            $shipment->provider = "Delhivery";
            $shipment->request_body = json_encode($shipmentDetails);
            $shipment->formatted_request = json_encode($FormattedData);

            $apiResponse = Http::withHeaders([
                'Accept' => 'application/json',
                'Authorization' => 'Token ' . $apiToken,
                'Content-Type' => 'application/json',
            ])->asForm()->post(config('services.delhivery.base_url') . config('services.delhivery.create_shipment_url'), $FormattedData);

            $responseData = $apiResponse->json();

            // return $responseData;
            $shipment->api_response = json_encode($apiResponse->json());

            if ($apiResponse->successful() && isset($responseData['success']) && $responseData['success'] === true) {

                $packages = $responseData['packages'] ?? [];

                if (!empty($packages)) {
                    $waybills = collect($packages)->pluck('waybill')->filter()->values()->all();
                    $shipment->tracking_no = json_encode($waybills);

                    // if (count($waybills) === 1) {
                    //     $shipment->tracking_no = json_encode($waybills[0]);
                    // } else {
                    //     $shipment->tracking_no = implode(',', $waybills);
                    // }

                    $shipment->save();

                    $labelResponse = $this->generateShippingLabels($waybills);
                    // dd($labelResponse);
                    $labelPackages = collect($labelResponse['packages'] ?? []);

                    $packageDetails = [];
                    foreach ($packages as $pkg) {
                        $labelData = $labelPackages->firstWhere('waybill', $pkg['waybill']);
                        $packageDetails[] = [
                            'waybill'     => $pkg['waybill'] ?? null,
                            'label_url'     => $labelData['pdf_download_link'] ?? null,
                            'reference'   => $pkg['refnum'] ?? null,
                            'status'      => $pkg['status'] ?? null,
                            'paymentMode' => $pkg['payment'] ?? null,
                            'serviceable' => $pkg['serviceable'] ?? null,
                        ];
                    }

                    return [
                        'status' => 'success',
                        'source' => 'Delhivery',
                        'message' => 'Shipment created successfully',
                        'labelUrl' => null,
                        'trackingNumber' => $waybills,
                        'upload_wbn' => $responseData['upload_wbn'] ?? null,
                        'shipmentDetails' => $packageDetails,
                    ];
                }
            }

            $shipment->save();

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => $responseData['rmk'] ?? 'Failed to create shipment',
                'data' => $apiResponse->json()
            ];
        } catch (\Exception $e) {
            Log::error('Delhivery Shipment API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Delhivery API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    private function generateShippingLabels(array $wbns): array
    {
        $token = config('services.delhivery.token');
        $baseUrl = config('services.delhivery.base_url');
        $labelUrl = config('services.delhivery.shipping_label_url');
        $results = [];

        try {
            $waybillString = implode(',', $wbns);
            $url = "{$baseUrl}{$labelUrl}?wbns={$waybillString}&pdf=true&pdf_size=4R";

            $response = Http::withHeaders([
                'Authorization' => "Token {$token}",
                'Accept'        => 'application/json',
            ])->get($url);

            if ($response->successful()) {
                $data = $response->json();

                if (!empty($data['packages'])) {
                    foreach ($data['packages'] as $pkg) {
                        $wbn = $pkg['wbn'] ?? null;
                        $pdfLink = $pkg['pdf_download_link'] ?? null;
                        $pdfBase64 = $pkg['pdf_encoding'] ?? null;

                        $pdfContent = Http::get($pdfLink)->body();

                        $fileName = 'label_' . $wbn . '.pdf';
                        Storage::disk('public')->put('shipping_labels/' . $fileName, $pdfContent);

                        $shortUrl = asset('storage/shipping_labels/' . $fileName);

                        $results[] = [
                            'waybill' => $wbn,
                            'pdf_download_link' => $shortUrl,
                            'message' => 'Label generated successfully'
                        ];
                    }
                } else {
                    $results[] = [
                        'error' => 'No packages found in response',
                    ];
                }
            } else {
                $results[] = [
                    'error' => 'API call failed with status: ' . $response->status(),
                ];
            }
        } catch (\Throwable $e) {
            $results[] = [
                'error' => $e->getMessage(),
            ];
        }

        return ['packages' => $results];
    }

    public function createPickup(array $pickupDetails): array
    {
        try {
            $payload = [
                'pickup_time' => $pickupDetails['closeTime'] ?? '11:00:00',
                'pickup_date' => date('Y-m-d', strtotime($pickupDetails['pickupDate'])) ?? date('Y-m-d'),
                'pickup_location' => $pickupDetails['pickupLocation']['contactName'],
                'expected_package_count' => $pickupDetails['expected_package_count'] ?? 1,
            ];

            $url = config('services.delhivery.base_url') . '' . config('services.delhivery.create_pickup_url');

            $pickup = new Pickup();
            $pickup->provider = "Delhivery";
            $pickup->request_body = json_encode($pickupDetails);
            $pickup->formatted_request = json_encode($payload);

            $apiResponse = Http::withHeaders([
                'Authorization' => 'Token ' . config('services.delhivery.token'),
                'Content-Type' => 'application/json',
            ])->post($url, $payload);

            $responseData = $apiResponse->json();
        //    dd($responseData);
            // return $responseData;
            $pickup->api_response = json_encode($apiResponse->json());

            if ($apiResponse->successful() && ($responseData['success'] ?? true) === true && isset($responseData['pickup_id'])) {
                $pickup->pickup_confirmation_code = $responseData['pickup_id'] ?? null;
                $pickup->pickup_location_code = $responseData['pickup_location_name'] ?? null;
                $pickup->save();

                return [
                    'status' => 'success',
                    'source' => 'Delhivery',
                    'message' => 'Pickup scheduled successfully',
                    'pickupConfirmationCode' => $responseData['pickup_id'],
                    'location' => $responseData['pickup_location_name'] ?? null,
                ];
            }

            $pickup->save();

            if (isset($responseData['error'])) {
                return [
                    'status' => 'error',
                    'source' => 'Delhivery',
                    'message' => $responseData['error']['message'] ?? 'Unknown error',
                    'data' => $responseData,
                ];
            }

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Failed to schedule pickup',
                'data' => $responseData,
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Pickup API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Delhivery API Exception: ' . $e->getMessage(),
                'data' => null,
            ];
        }
    }

    public function cancelPickup(array $pickupDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $FormattedData = FedExAdapter::formatCancelPickupRequest($pickupDetails);

            $pickup = Pickup::where(['provider' => "Fedex", 'is_canceled' => 0, 'pickup_confirmation_code' => $pickupDetails['pickupConfirmationCode'], 'pickup_location_code' => $pickupDetails['location']])->first();

            $pickup->cancel_request = json_encode($FormattedData);

            $apiResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.cancel_pickup_url'), $FormattedData);

            $responseData = $apiResponse->json();

            // return $responseData;
            $pickup->cancel_response = json_encode($apiResponse->json());

            if ($apiResponse->successful() && isset($responseData['output']['pickupConfirmationCode'])) {
                $pickup->is_canceled = 1;
                $pickup->save();

                return [
                    'status' => 'success',
                    'source' => 'FedEx',
                    'message' => $responseData['output']['cancelConfirmationMessage']
                        ?? 'Pickup cancelled successfully',
                    'pickupConfirmationCode' => $responseData['output']['pickupConfirmationCode'],
                ];
            }

            $pickup->save();

            if (isset($responseData['errors'])) {
                return FedExAdapter::formatErrorResponse($responseData);
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to cancel pickup',
                'data' => $responseData,
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Pickup Cancellation API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null,
            ];
        }
    }

    public function returnShipment(array $shipmentDetails): array
    {
        try {
            $apiToken = config('services.delhivery.token');

            $FormattedData = DelhiveryAdapter::formatCreateShipmentRequest($shipmentDetails);
            // dd($FormattedData);
            $shipment = new Shipment();
            $shipment->is_returned = 1;
            $shipment->provider = "Delhivery";
            $shipment->request_body = json_encode($shipmentDetails);
            $shipment->formatted_request = json_encode($FormattedData);

            $apiResponse = Http::withHeaders([
                'Accept' => 'application/json',
                'Authorization' => 'Token ' . $apiToken,
                'Content-Type' => 'application/json',
            ])->asForm()->post(config('services.delhivery.base_url') . config('services.delhivery.create_shipment_url'), $FormattedData);

            $responseData = $apiResponse->json();

            // return $responseData;
            $shipment->api_response = json_encode($apiResponse->json());

            if ($apiResponse->successful() && isset($responseData['success']) && $responseData['success'] === true) {

                $packages = $responseData['packages'] ?? [];

                if (!empty($packages)) {
                    $waybills = collect($packages)->pluck('waybill')->filter()->values()->all();
                    $shipment->tracking_no = json_encode($waybills);

                    $shipment->save();

                    $labelResponse = $this->generateShippingLabels($waybills);
                    // dd($labelResponse);
                    $labelPackages = collect($labelResponse['packages'] ?? []);

                    $packageDetails = [];
                    foreach ($packages as $pkg) {
                        $labelData = $labelPackages->firstWhere('waybill', $pkg['waybill']);
                        $packageDetails[] = [
                            'waybill'     => $pkg['waybill'] ?? null,
                            'label_url'     => $labelData['pdf_download_link'] ?? null,
                            'reference'   => $pkg['refnum'] ?? null,
                            'status'      => $pkg['status'] ?? null,
                            'paymentMode' => $pkg['payment'] ?? null,
                            'serviceable' => $pkg['serviceable'] ?? null,
                        ];
                    }

                    return [
                        'status' => 'success',
                        'source' => 'Delhivery',
                        'message' => 'Shipment created successfully',
                        'labelUrl' => null,
                        'trackingNumber' => $waybills,
                        'upload_wbn' => $responseData['upload_wbn'] ?? null,
                        'shipmentDetails' => $packageDetails,
                    ];
                }
            }

            $shipment->save();

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => $responseData['rmk'] ?? 'Failed to create shipment',
                'data' => $apiResponse->json()
            ];
        } catch (\Exception $e) {
            Log::error('Delhivery Return Shipment API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Delhivery API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    public function cancelShipment(array $shipmentDetails): array
    {
        try {
            $baseUrl = config('services.delhivery.base_url');
            $endpoint = config('services.delhivery.cancel_shipment_url');
            $url = "{$baseUrl}{$endpoint}";

            $token = config('services.delhivery.token');

            $payload = [
                'waybill' => $shipmentDetails['tracking_number'],
                'cancellation' => 'true',
            ];

            $shipment = Shipment::whereJsonContains('tracking_no', $shipmentDetails['tracking_number'])->where('is_canceled', 0)->first();

            if ($shipment) {
                $apiResponse = Http::withHeaders([
                    'Authorization' => 'Token ' . $token,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])->post($url, $payload);

                $responseBody = $apiResponse->json();


                $shipment->cancel_request = json_encode($payload);
                $shipment->cancel_response = json_encode($responseBody);

                if ($apiResponse->successful() && ($responseBody['status'] ?? false) === true) {
                    $shipment->is_canceled = 1;
                }
                $shipment->save();

                if ($apiResponse->successful() && ($responseBody['status'] ?? false) === true) {
                    return [
                        'status' => 'success',
                        'source' => 'Delhivery',
                        'message' => $responseBody['remark'] ?? 'Shipment cancelled successfully',
                        'data' => [
                            'cancelledShipment' => true,
                            'cancelledHistory' => $responseBody['waybill'] ?? null,
                            'transactionId' => $responseBody['order_id'] ?? null,
                        ],
                    ];
                }
            }

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => $responseBody['error']
                    ?? 'Shipment cancellation failed (possibly already cancelled or invalid tracking number)',
                'data' => [
                    'cancelledShipment' => false,
                    'cancelledHistory' => $responseBody['waybill'] ?? false,
                    'transactionId' => $responseBody['order_id'] ?? null,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Delhivery Shipment Cancellation API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'Delhivery',
                'message' => 'Delhivery API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    private function buildPostalCode($base, $addon)
    {
        return $addon ? "$base-$addon" : $base;
    }
}
