<?php

namespace App\Adapters\Shipping;

use Carbon\Carbon;
use Illuminate\Support\Facades\Http;

class DelhiveryAdapter
{
    public static function formatShippingCostParams(array $shortRequest): array
    {
        $cgm = self::convertToGrams($shortRequest['weight_value'], $shortRequest['weight_unit']);

        return [
            'md'     => $shortRequest['md'] ?? 'S',                 // Express / Surface
            'ss'     => $shortRequest['ss'] ?? 'Delivered',         // Delivered / RTO / DTO
            'd_pin'  => $shortRequest['recipient_postal_code'],     // Destination pincode
            'o_pin'  => $shortRequest['shipper_postal_code'],       // Origin pincode
            'cgm'    => $cgm,                                       // Chargeable weight (grams)
            'pt'     => $shortRequest['pt'] ?? 'Pre-paid',          // Payment type (Pre-paid / COD)
        ];
    }

    private static function convertToGrams(float $value, string $unit = 'kg'): float
    {
        switch (strtolower($unit)) {
            case 'kg':
                return $value * 1000;
            case 'lb':
            case 'pound':
                return $value * 453.592;
            case 'gm':
            default:
                return $value;
        }
    }

    public static function formatCreateShipmentRequest(array $input): array
    {
        $packageCount = (int) $input['shipping']['no_of_packages'];
        $shipments = [];

        $waybills = self::generateWaybills($packageCount);

        $masterWaybill = trim($waybills[0]) ?? null;

        foreach ($input['packages'] as $index => $package) {
            $waybillNumber = trim($waybills[$index]) ?? null;

            $shipment = [
                "name" => $input['recipient']['name'],
                "add" => implode(', ', $input['recipient']['address']),
                "pin" => $input['recipient']['postal_code'],
                "city" => $input['recipient']['city'],
                "state" => $input['recipient']['state'],
                "country" => $input['recipient']['country'],
                "phone" => $input['recipient']['phone'],

                "seller_name" => $input['shipper']['name'],
                "seller_add" => implode(', ', $input['shipper']['address']),
                "seller_inv" => $package['customer_references'][0]['value'] ?? '',

                "shipment_length" => $package['dimensions']['length'],
                "shipment_width"  => $package['dimensions']['width'],
                "shipment_height" => $package['dimensions']['height'],
                "weight" => $package['weight'],

                "order" => $package['customer_references'][0]['value'] ?? '',
                "products_desc" => implode(', ', array_column($input['customs'], 'description')),
                "order_date" => $input['shipment']['date'],
                "quantity" => $input['customs'][0]['quantity'] ?? 1,

                "waybill" => $waybillNumber,

                "payment_mode" => $input['shipping']['payment_mode'],
                "shipping_mode" => $input['shipping']['shipping_mode'],
                "ewbn" => $input['shipping']['eway_bill_no'],
                "hsn_code" => $input['shipping']['hsn_code'],
                "cod_amount" => $input['shipping']['cod_amount'],
                "address_type" => $input['shipping']['address_type'],
                "fragile_shipment" => $input['shipping']['fragile_shipment'],
                "dangerous_good" => $input['shipping']['dangerous_good'],
                "total_amount" => $input['shipping']['total_amount'],
            ];

            // If more than one package → add MPS fields
            if ($packageCount > 1) {
                $shipment['mps_amount'] = $input['shipping']['cod_amount'];
                $shipment['mps_children'] = $packageCount;
                $shipment['master_id'] = $masterWaybill;
                $shipment['shipment_type'] = 'MPS';
            }

            $shipments[] = $shipment;
        }

        return [
            "format" => "json",
            'data' => json_encode([
                'shipments' => $shipments,
                'pickup_location' => [
                    'name' => 'TVIKLA SURFACE',
                ],
            ]),
        ];
    }

    private static function generateWaybills(int $count): array
    {
        $token = config('services.delhivery.token');
        // $url = "https://staging-express.delhivery.com/waybill/api/bulk/json/?token={$token}&count={$count}";
        $singlewaybill_url = config('services.delhivery.base_url') . config('services.delhivery.single_waybill_url');
        $multiplewaybill_url = config('services.delhivery.base_url') . config('services.delhivery.multiple_waybill_url');

        if ($count > 1) {
            $url = "{$multiplewaybill_url}?token={$token}&count={$count}";
        } else {
            $url = "{$singlewaybill_url}?token={$token}";
        }

        $response = Http::withHeaders([
            'Accept' => 'application/json',
        ])->get($url);

        if ($response->successful()) {
            $body = trim($response->json());

            $waybills = array_filter(array_map('trim', explode(',', $body)));

            if (!empty($waybills)) {
                return array_values($waybills); // ensure numeric array
            }
        }

        throw new \Exception('Failed to generate waybill numbers from Delhivery');
    }

    public static function simplifyTrackingResponse(array $tracking): array
    {
        $shipments = $tracking['ShipmentData'] ?? [];
        $simplified = [];

        foreach ($shipments as $shipmentItem) {
            $shipment = $shipmentItem['Shipment'] ?? [];

            $status = $shipment['Status'] ?? [];
            $scans = $shipment['Scans'][0]['ScanDetail'] ?? [];

            $simplified[] = [
                'trackingNumber'   => $shipment['AWB'] ?? null,
                'service'          => $shipment['OrderType'] ?? null,
                'status'           => $status['Status'] ?? null,
                'lastScanDate'     => $scans['ScanDateTime'] ?? null,
                'lastScanStatus'   => $scans['Scan'] ?? null,
                'lastScanCity'     => $status['StatusLocation'] ?? null,
                'weightKG'         => $shipment['ChargedWeight'] ?? null,
                'originCity'       => $shipment['Origin'] ?? null,
                'destinationCity'  => $shipment['Destination'] ?? null,
                'shipperCity'      => $shipment['PickupLocation'] ?? null,
                'recipientCity'    => $shipment['Consignee']['City'] ?? null,
                'invoiceAmount'    => $shipment['InvoiceAmount'] ?? null,
                'codAmount'        => $shipment['CODAmount'] ?? null,
                'referenceNumber'  => $shipment['ReferenceNo'] ?? null,
                'pickupDate'       => $shipment['PickUpDate'] ?? null,
                'deliveryDate'     => $shipment['DeliveryDate'] ?? null,
                'expectedDelivery' => $shipment['ExpectedDeliveryDate'] ?? null,
            ];
        }

        return $simplified;
    }

    public static function formatReturnShipmentRequest(array $input): array
    {
        $packageCount = (int) $input['shipping']['no_of_packages'];
        $shipments = [];

        $waybills = self::generateWaybills($packageCount);

        $masterWaybill = trim($waybills[0]) ?? null;

        foreach ($input['packages'] as $index => $package) {
            $waybillNumber = trim($waybills[$index]) ?? null;

            $shipment = [
                "name" => $input['recipient']['name'],
                "add" => implode(', ', $input['recipient']['address']),
                "pin" => $input['recipient']['postal_code'],
                "city" => $input['recipient']['city'],
                "state" => $input['recipient']['state'],
                "country" => $input['recipient']['country'],
                "phone" => $input['recipient']['phone'],

                "return_name" => $input['shipper']['name'],
                "return_add" => $input['shipping']['registered_warehouse_name'],
                "return_address" => implode(', ', $input['shipper']['address']),
                "return_city" => $input['shipper']['city'],
                "return_phone" => $input['shipper']['phone'],
                "return_state" => $input['shipper']['state'],
                "return_country" => $input['shipper']['country'],
                "return_pin" => $input['shipper']['postal_code'],
                // "seller_inv" => $package['customer_references'][0]['value'] ?? '',

                "shipment_length" => $package['dimensions']['length'],
                "shipment_width"  => $package['dimensions']['width'],
                "shipment_height" => $package['dimensions']['height'],
                "weight" => $package['weight'],

                "order" => $package['customer_references'][0]['value'] ?? '',
                "products_desc" => implode(', ', array_column($input['customs'], 'description')),
                "order_date" => $input['shipment']['date'],
                "quantity" => $input['customs'][0]['quantity'] ?? 1,

                "waybill" => $waybillNumber,

                "payment_mode" => $input['shipping']['payment_mode'],
                "shipping_mode" => $input['shipping']['shipping_mode'],
                "ewbn" => $input['shipping']['eway_bill_no'],
                "hsn_code" => $input['shipping']['hsn_code'],
                "cod_amount" => $input['shipping']['cod_amount'],
                "address_type" => $input['shipping']['address_type'],
                "fragile_shipment" => $input['shipping']['fragile_shipment'],
                "dangerous_good" => $input['shipping']['dangerous_good'],
                "total_amount" => $input['shipping']['total_amount'],
            ];

            // If more than one package → add MPS fields
            if ($packageCount > 1) {
                $shipment['mps_amount'] = $input['shipping']['cod_amount'];
                $shipment['mps_children'] = $packageCount;
                $shipment['master_id'] = $masterWaybill;
                $shipment['shipment_type'] = 'MPS';
            }

            $shipments[] = $shipment;
        }

        return [
            "format" => "json",
            'data' => json_encode([
                'shipments' => $shipments,
                'pickup_location' => [
                    'name' => $input['shipping']['registered_warehouse_name'],
                ],
            ]),
        ];
    }
}
