<?php

namespace App\Http\Controllers\Api;

use App\Adapters\Shipping\FedexAdapter;
use App\Adapters\Shipping\UpsAdapter;
use App\Http\Controllers\Api\BaseController;
use App\Models\Shipment;
use Illuminate\Http\Request;
use App\Services\Logistics\LogisticsFactory;
use App\Traits\ResponseTrait;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class LogisticsController extends BaseController
{
    public function validateAddress(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'address.company_name'    => 'nullable|string|max:255',
            'address.building_name'   => 'nullable|string|max:255',
            'address.street'          => 'required|array|min:1|max:3',
            'address.street.*'        => 'required|string|max:255',
            'address.city'            => 'required|string|max:255',
            'address.state'           => 'required|string|max:10',
            'address.postal_code'     => 'required|string|max:10',
            'address.postal_code_ext' => 'nullable|string|max:10',
            'address.country'         => 'required|string|size:2',
        ]);

        // if ($validator->fails()) {
        //     return response()->json([
        //         'status' => 'error',
        //         'message' => 'Validation failed',
        //         'errors' => $validator->errors(),
        //         'status_code' => 422,
        //     ], 422);
        // }
        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->input('address');

        $shippingOptions = LogisticsFactory::availablityCheck($input);

        $successCount = collect($shippingOptions)
            ->filter(fn($option) => isset($option['status']) && $option['status'] === 'success')
            ->count();

        if ($successCount > 0) {
            return response()->json([
                'status' => 'success',
                'message' => $successCount < count($shippingOptions)
                    ? 'Address validation completed with partial results'
                    : 'Address validated successfully via all providers',
                'data' => $shippingOptions,
                'status_code' => 200,
            ]);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'All address validation providers failed',
                'data' => $shippingOptions,
                'status_code' => 422
            ], 422);
        }
    }

    public function getAllShippingCosts(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shipper_postal_code' => 'required|numeric',
            'shipper_country_code' => 'required|string|size:2',
            'recipient_postal_code' => 'required|numeric',
            'recipient_country_code' => 'required|string|size:2',
            'pickup_type' => 'required|string',
            'currency' => 'required|string|size:3',
            'weight_value' => 'required|numeric|min:0.1',
            'weight_unit' => 'required|string|in:LB,KG',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $validated = $request->all();

        $providers = LogisticsFactory::getProviderNames();

        if (!(strtoupper($validated['shipper_country_code']) === 'IN' && strtoupper($validated['recipient_country_code']) === 'IN')) {
            $providers = array_filter($providers, fn($p) => !in_array(strtolower($p), ['dhl', 'delhivery']));
        }

        try {
            $shippingOptions = LogisticsFactory::getAllShippingCosts($validated, $providers);

            if (empty($shippingOptions)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No shipping options available',
                    'data' => null,
                    'status_code' => 404,
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Shipping costs retrieved successfully',
                'data' => $shippingOptions,
                'status_code' => 200,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred while fetching shipping costs',
                'error' => $e->getMessage(),
                'status_code' => 500,
            ], 500);
        }
    }

    public function pickupAvailability(Request $request)
    {
        $v = Validator::make($request->all(), [
            'pickupAddress.streetLines'         => 'required|array|min:1|max:2',
            'pickupAddress.streetLines.*'       => 'required|string|max:255',
            'pickupAddress.city'                => 'required|string|max:255',
            'pickupAddress.stateOrProvinceCode' => 'required|string|size:2',
            'pickupAddress.postalCode'          => 'required|string|max:10',
            'pickupAddress.countryCode'         => 'required|string|size:2',

            'carrierCode'         => 'required|string',
            'pickupRequestType'   => 'required|array|min:1',
            'pickupRequestType.*' => 'required|string',
            'localityType'        => 'required|string',

            'dispatchDate'        => 'required|date|after_or_equal:today',
            'packageReadyTime'    => 'required|date_format:H:i:s',
            'customerCloseTime'   => 'required|date_format:H:i:s',
            'pickupType'          => 'required|string',

            'carriers'            => 'required|array|min:1',
            'carriers.*'          => 'required|string',

            'countryRelationship' => 'required|string',
        ]);

        if ($v->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $v->errors(),
                'status_code' => 422,
            ], 422);
        }

        $data = $request->only([
            'pickupAddress',
            'carrierCode',
            'pickupRequestType',
            'localityType',
            'dispatchDate',
            'packageReadyTime',
            'customerCloseTime',
            'pickupType',
            'carriers',
            'countryRelationship',
        ]);

        $options = LogisticsFactory::checkPickupAvailability($data);

        $success = collect($options)
            ->filter(fn($o) => isset($o['availableSlots']) && count($o['availableSlots']) > 0)
            ->count();

        return response()->json([
            'status' => $success ? 'success' : 'error',
            'message' => $success
                ? 'Pickup available'
                : 'No pickup availability',
            'data' => $options,
            'status_code' => $success ? 200 : 400,
        ], $success ? 200 : 422);
    }

    public function trackShipment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'provider'          => 'required|string',
            'tracking_number'   => 'required|string|max:50',
            'show_full_history' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::TrackShipment($input, $input['provider'] ?? 'fedex');

        $response = is_array($apiResponse) && isset($apiResponse[0])
            ? $apiResponse[0]
            : $apiResponse;

        return response()->json([
            'status'      => $response['status'] ?? 'error',
            'provider'    => $response['source'] ?? $input['provider'],
            'message'     => $response['message'] ?? ($response['status'] === 'success'
                ? 'Shipment tracking successfully'
                : 'Shipment tracking failed'),
            'data'        => $response['data'] ?? null,
            'status_code' => ($response['status'] === 'success') ? 200 : 400,
        ], ($response['status'] === 'success') ? 200 : 400);

        return response()->json($response, $response['status'] === 'success' ? 200 : 400);
    }

    public function createShipment(Request $request)
    {
        // return $request->all();
        $request->merge([
            'shipping' => array_merge($request->input('shipping', []), [
                'provider' => strtolower($request->input('shipping.provider'))
            ])
        ]);
        $validator = Validator::make($request->all(), [
            'shipping.provider' => 'required|string|max:255',
            'shipping.payment_mode' => 'nullable|required_if:shipping.provider,delhivery|string|max:50',
            'shipping.shipping_mode' => 'nullable|required_if:shipping.provider,delhivery|string|max:50',
            'shipping.registered_warehouse_name' => 'nullable|required_if:shipping.provider,delhivery|string',
            'shipping.total_amount' => 'nullable|numeric|min:0',
            'shipping.eway_bill_no' => [
                'nullable',
                'string',
                'max:255',
                function ($attribute, $value, $fail) use ($request) {
                    $provider = strtolower($request->input('shipping.provider'));
                    $totalAmount = $request->input('shipping.total_amount');

                    if ($provider === 'delhivery' && $totalAmount >= 50000 && empty($value)) {
                        $fail('The eWay bill number is required for Delhivery shipments when total amount is 50,000 or more.');
                    }
                },
            ],
            'shipping.hsn_code' => 'nullable|string|max:255',
            'shipping.cod_amount' => 'nullable|numeric|min:0',
            'shipping.address_type' => 'nullable|string|max:50',
            'shipping.fragile_shipment' => 'nullable|string|max:50',
            'shipping.dangerous_good' => 'nullable|string|max:50',
            'shipping.no_of_packages' => 'required_if:shipping.provider,delhivery|numeric|min:1',

            'shipper.name' => 'required|string|max:255',
            'shipper.phone' => 'required|string|max:20',
            'shipper.address' => 'required',
            'shipper.city' => 'required|string',
            'shipper.state' => 'required|string|max:10',
            'shipper.postal_code' => 'required|string',
            'shipper.country' => 'required|string|size:2',

            'recipient.name' => 'required|string|max:255',
            'recipient.phone' => 'required|string|max:20',
            'recipient.address' => 'required',
            'recipient.city' => 'required|string',
            'recipient.state' => 'required|string|max:10',
            'recipient.postal_code' => 'required|string',
            'recipient.country' => 'required|string|size:2',

            'packages' => 'required|array|min:1',
            'packages.*.weight' => 'required|numeric|min:0.01',
            'packages.*.weight_unit' => 'required|string|size:2',
            'packages.*.dimensions_unit' => 'required|string|size:2',
            'packages.*.dimensions.length' => 'required|numeric|min:1',
            'packages.*.dimensions.width' => 'required|numeric|min:1',
            'packages.*.dimensions.height' => 'required|numeric|min:1',

            'shipment.date' => 'required|date',
            'shipment.service_type' => 'required_if:shipping.provider,fedex|nullable|string',
            'shipment.packaging_type' => 'required_if:shipping.provider,fedex|nullable|string',
            'shipment.pickup_type' => 'required_if:shipping.provider,fedex|nullable|string',

            'customs' => 'required|array|min:1',
            'customs.*.description' => 'required|string',
            'customs.*.quantity' => 'required|integer|min:1',
            'customs.*.price_value' => 'required|numeric|min:0.01',
            'customs.*.currency' => 'required|string|size:3',
            'customs.*.country_of_manufacture' => 'required|string|size:2',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors,
                'provider' => $request->input('shipping.provider')
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::createShipment($input, $input['shipping']['provider'] ?? 'fedex');
        // return $apiResponse;

        $apiResponse = $apiResponse[0] ?? [];

        if (!empty($apiResponse) && $apiResponse['status'] === 'success') {
            return response()->json([
                'status' => 'success',
                'provider' => $apiResponse['source'],
                'message' => $apiResponse['message'],
                'data' => [
                    'details' => $apiResponse['shipmentDetails'] ?? [],
                ],
                'status_code' => 200,
            ], 200);
        }

        if (!empty($apiResponse['errors'])) {
            return response()->json($apiResponse, 422);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $apiResponse['source'],
            'message' => $apiResponse['message'] ?? 'Shipment creation failed',
            'data' => $apiResponse['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function createPickup(Request $request)
    {
        $request->merge([
            'shippingProvider' => strtolower($request->input('shippingProvider')),
        ]);

        $validator = Validator::make($request->all(), [
            'shippingProvider' => 'required|string',
            'carrierCode'      => 'nullable|required_if:shippingProvider,fedex|string',
            'pickupLocation.contactName' => 'required|string|max:255',
            'pickupLocation.phone'       => 'nullable|required_if:shippingProvider,fedex|string|max:20',
            'pickupLocation.company'     => 'nullable|string|max:255',
            'pickupLocation.address.streetLines' => 'nullable|required_if:shippingProvider,fedex|array',
            'pickupLocation.address.streetLines.*' => 'nullable|required_if:shippingProvider,fedex|string|max:255',
            'pickupLocation.address.city'          => 'nullable|required_if:shippingProvider,fedex|string|max:255',
            'pickupLocation.address.state'         => 'nullable|required_if:shippingProvider,fedex|string|max:255',
            'pickupLocation.address.postalCode'    => 'nullable|required_if:shippingProvider,fedex|string|max:20',
            'pickupLocation.address.country'       => 'nullable|required_if:shippingProvider,fedex|string|size:2',
            'pickupDate'    => 'required|date',
            'closeTime'     => 'required|string',
            'packageCount'  => 'required|integer|min:1',
            'totalWeight.units' => 'nullable|required_if:shippingProvider,fedex|string|in:KG,LB',
            'totalWeight.value' => 'nullable|required_if:shippingProvider,fedex|numeric|min:0.1',
            'packageLocation'   => 'nullable|required_if:shippingProvider,fedex|string|max:50',
            'trackingNumber'    => 'nullable|string|max:50',
            'remarks'           => 'nullable|string|max:500',
            'deliveryInstructions'           => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors,
                'provider' => $request->input('shippingProvider')
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::createPickup($input, $input['shippingProvider'] ?? 'fedex');
        // return $apiResponse;

        $pickupResponse = $apiResponse[0] ?? [];

        if (!empty($pickupResponse) && $pickupResponse['status'] === 'success') {
            return response()->json([
                'status' => 'success',
                'provider' => $pickupResponse['source'],
                'message' => $pickupResponse['message'],
                'data' => [
                    'pickupConfirmationCode' => $pickupResponse['pickupConfirmationCode'] ?? null,
                    'location' => $pickupResponse['location'] ?? null,
                ],
                'status_code' => 200,
            ], 200);
        }

        if (!empty($pickupResponse['errors'])) {
            return response()->json($pickupResponse, 422);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $pickupResponse['source'] ?? 'FedEx',
            'message' => $pickupResponse['message'] ?? 'Pickup scheduling failed',
            'data' => $pickupResponse['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function cancelPickup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shippingProvider' => 'required|string',
            'pickupConfirmationCode' => 'required|string|max:50',
            'carrierCode'            => 'required|string', // FedEx Express / FedEx Ground
            'scheduledDate'          => 'required|date|after_or_equal:today',
            'location'               => 'required|string|max:10',
            'remarks'                => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::cancelPickup($input, $input['shippingProvider'] ?? 'fedex');
        // return $apiResponse;

        $pickupResponse = $apiResponse[0] ?? [];

        if (!empty($pickupResponse) && $pickupResponse['status'] === 'success') {
            return response()->json([
                'status' => 'success',
                'provider' => $pickupResponse['source'],
                'message' => $pickupResponse['message'],
                'data' => [
                    'pickupConfirmationCode' => $pickupResponse['pickupConfirmationCode'] ?? null,
                ],
                'status_code' => 200,
            ], 200);
        }

        if (!empty($pickupResponse['errors'])) {
            return response()->json($pickupResponse, 422);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $pickupResponse['source'] ?? 'FedEx',
            'message' => $pickupResponse['message'] ?? 'Pickup scheduling failed',
            'data' => $pickupResponse['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function returnShipment(Request $request)
    {
        $request->merge([
            'shipping' => array_merge($request->input('shipping', []), [
                'provider' => strtolower($request->input('shipping.provider'))
            ])
        ]);
        $validator = Validator::make($request->all(), [
            'shipping.provider' => 'required|string|max:255',
            'shipping.payment_mode' => [
                'required_if:shipping.provider,delhivery',
                'in:Pickup,REPL',
            ],
            'shipping.shipping_mode' => 'nullable|required_if:shipping.provider,delhivery|string|max:50',
            'shipping.registered_warehouse_name' => 'nullable|required_if:shipping.provider,delhivery|string',
            'reference' => 'nullable|required_if:shipping.provider,fedex|string',
            'shipping.total_amount' => 'nullable|numeric|min:0',
            'shipping.eway_bill_no' => [
                'nullable',
                'string',
                'max:255',
                function ($attribute, $value, $fail) use ($request) {
                    $provider = strtolower($request->input('shipping.provider'));
                    $totalAmount = $request->input('shipping.total_amount');

                    if ($provider === 'delhivery' && $totalAmount >= 50000 && empty($value)) {
                        $fail('The eWay bill number is required for Delhivery shipments when total amount is 50,000 or more.');
                    }
                },
            ],
            'shipping.hsn_code' => 'nullable|string|max:255',
            'shipping.cod_amount' => 'nullable|numeric|min:0',
            'shipping.address_type' => 'nullable|string|max:50',
            'shipping.fragile_shipment' => 'nullable|string|max:50',
            'shipping.dangerous_good' => 'nullable|string|max:50',
            'shipping.no_of_packages' => 'nullable|required_if:shipping.provider,delhivery|numeric|min:1',

            'shipper.name' => 'required|string|max:255',
            'shipper.phone' => 'required|string|max:20',
            'shipper.address' => 'required',
            'shipper.city' => 'required|string',
            'shipper.state' => 'required|string|max:10',
            'shipper.postal_code' => 'required|string',
            'shipper.country' => 'required|string|size:2',

            'recipient.name' => 'required|string|max:255',
            'recipient.phone' => 'required|string|max:20',
            'recipient.address' => 'required',
            'recipient.city' => 'required|string',
            'recipient.state' => 'required|string|max:10',
            'recipient.postal_code' => 'required|string',
            'recipient.country' => 'required|string|size:2',

            'packages' => 'required|array|min:1',
            'packages.*.weight' => 'required|numeric|min:0.01',
            'packages.*.weight_unit' => 'required|string|size:2',
            'packages.*.dimensions_unit' => 'required|string|size:2',
            'packages.*.dimensions.length' => 'required|numeric|min:1',
            'packages.*.dimensions.width' => 'required|numeric|min:1',
            'packages.*.dimensions.height' => 'required|numeric|min:1',

            'shipment.date' => 'required|date',
            'shipment.service_type' => 'required_if:shipping.provider,fedex|nullable|string',
            'shipment.packaging_type' => 'required_if:shipping.provider,fedex|nullable|string',
            'shipment.pickup_type' => 'required_if:shipping.provider,fedex|nullable|string',

            'customs' => 'required|array|min:1',
            'customs.*.description' => 'required|string',
            'customs.*.quantity' => 'required|integer|min:1',
            'customs.*.price_value' => 'required|numeric|min:0.01',
            'customs.*.currency' => 'required|string|size:3',
            'customs.*.country_of_manufacture' => 'required|string|size:2',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::returnShipment($input, $input['shipping']['provider'] ?? 'fedex');
        // return $apiResponse;

        $apiResponse = $apiResponse[0] ?? [];

        if (!empty($apiResponse) && $apiResponse['status'] === 'success') {
            return response()->json([
                'status' => 'success',
                'provider' => $apiResponse['source'],
                'message' => $apiResponse['message'],
                'data' => [
                    'details' => $apiResponse['shipmentDetails'] ?? [],
                ],
                'status_code' => 200,
            ], 200);
        }

        if (!empty($apiResponse['errors'])) {
            return response()->json($apiResponse, 422);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $apiResponse['source'],
            'message' => $apiResponse['message'] ?? 'Shipment creation failed',
            'data' => $apiResponse['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function returnShipment_old(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shipping.provider' => 'required|string|max:255',

            'shipper.name' => 'required|string|max:255',
            'shipper.phone' => 'required|string|max:20',
            'shipper.address' => 'required',
            'shipper.city' => 'required|string',
            'shipper.state' => 'required|string|max:10',
            'shipper.postal_code' => 'required|string',
            'shipper.country' => 'required|string|size:2',

            'recipient.name' => 'required|string|max:255',
            'recipient.phone' => 'required|string|max:20',
            'recipient.address' => 'required',
            'recipient.city' => 'required|string',
            'recipient.state' => 'required|string|max:10',
            'recipient.postal_code' => 'required|string',
            'recipient.country' => 'required|string|size:2',

            'package.weight_unit' => 'required|string|size:2',
            'package.dimensions_unit' => 'required|string|size:2',

            'package.weight' => 'required|numeric|min:0.01',
            'package.dimensions.length' => 'required|numeric|min:1',
            'package.dimensions.width' => 'required|numeric|min:1',
            'package.dimensions.height' => 'required|numeric|min:1',

            'shipment.date' => 'required|date',
            'shipment.service_type' => 'required|string',
            'shipment.packaging_type' => 'required|string',
            'shipment.pickup_type' => 'required|string',

            'customs.description' => 'required|string',
            'customs.quantity' => 'required|integer|min:1',
            'customs.value_usd' => 'required|numeric|min:0.01',
            'customs.currency' => 'required|string|size:3',
            'customs.country_of_manufacture' => 'required|string|size:2',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $apiResponse = LogisticsFactory::returnShipment($input, $input['shipping']['provider'] ?? 'fedex');
        // return $apiResponse;

        $apiResponse = $apiResponse[0] ?? [];

        if (!empty($apiResponse) && $apiResponse['status'] === 'success') {
            return response()->json([
                'status' => 'success',
                'provider' => $apiResponse['source'],
                'message' => $apiResponse['message'],
                'data' => [
                    // 'trackingNumber' => $apiResponse['trackingNumber'] ?? null,
                    // 'labelUrl' => $apiResponse['labelUrl'] ?? null,
                    'details' => $apiResponse['shipmentDetails'] ?? [],
                ],
                'status_code' => 200,
            ], 200);
        }

        if (!empty($apiResponse['errors'])) {
            return response()->json($apiResponse, 422);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $apiResponse['source'],
            'message' => $apiResponse['message'] ?? 'Shipment creation failed',
            'data' => $apiResponse['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function cancelShipment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'provider' => 'required|string',
            'tracking_number' => 'required|string',
            'sender_country' => 'required|string|size:2',
        ]);

        if ($validator->fails()) {
            $rawErrors = $validator->errors()->toArray();

            $formatted = FedExAdapter::formatValidationErrorResponse([
                'errors' => $rawErrors
            ]);

            return response()->json($formatted, 422);
        }

        $input = $request->all();

        $provider = strtolower($request->provider);

        $apiResponse = LogisticsFactory::cancelShipment($input, $provider);
        $response = $apiResponse[0];

        // return $apiResponse;

        if (
            !empty($apiResponse) &&
            isset($apiResponse[0]['status']) &&
            $apiResponse[0]['status'] === 'success'
        ) {
            return response()->json([
                'status' => 'success',
                'provider' => $response['source'],
                'message' => $response['message'] ?? 'Shipment cancelled successfully',
                'data' => $response['data'] ?? [],
                'status_code' => 200,
            ], 200);
        }

        return response()->json([
            'status' => 'error',
            'provider' => $response['source'],
            'message' => $response['message'] ?? 'Shipment cancellation failed',
            'data' => $response['data'] ?? null,
            'status_code' => 400,
        ], 400);
    }

    public function availablityCheck(Request $request)
    {

        $validated = $request->validate([
            'address.streetLines' => 'required|array|min:1|max:2',
            'address.streetLines.*' => 'required|string|max:100',
            'address.city' => 'required|string|max:100',
            'address.stateOrProvinceCode' => 'required|string|size:2',
            'address.postalCode' => 'required|string|max:20',
            'address.countryCode' => 'required|string|size:2',
        ], [], [
            'address.streetLines' => 'address.streetLines',
            'address.streetLines.*' => 'each street line',
            'address.city' => 'address.city',
            'address.stateOrProvinceCode' => 'address.stateOrProvinceCode',
            'address.postalCode' => 'address.postalCode',
            'address.countryCode' => 'address.countryCode',
        ]);

        $shippingOptions = LogisticsFactory::availablityCheck($validated['address']);

        if (empty($shippingOptions)) {
            return response()->json([
                'status' => 'error',
                'message' => 'No shipping options available',
                'data' => null,
                'status_code' => 404,
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Shipping costs retrieved successfully',
            'data' => $shippingOptions,
            'status_code' => 200,
        ], 200);
    }

    public function createDelhiveryWarehouse(Request $request)
    {
        $validated = $request->validate([
            'phone'           => 'required|string|max:15',
            'city'            => 'required|string',
            'name'            => 'required|string',
            'pin'             => 'required|digits_between:4,10',
            'address'         => 'required|string',
            'country'         => 'nullable|string',
            'email'           => 'nullable|email',
            'registered_name' => 'nullable|string',
            'return_address'  => 'required|string',
            'return_pin'      => 'required|digits_between:4,10',
            'return_city'     => 'nullable|string',
            'return_state'    => 'nullable|string',
            'return_country'  => 'nullable|string',
        ]);

        try {
            $baseUrl = config('services.delhivery.base_url');
            $warehouseUrl = config('services.delhivery.create_warehouse_url');
            $url = "{$baseUrl}{$warehouseUrl}";
            $token = config('services.delhivery.token');

            $response = Http::withHeaders([
                'Authorization' => 'Token ' . $token,
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ])->post($url, $validated);

            if ($response->successful()) {
                return response()->json([
                    'status'  => true,
                    'message' => 'Warehouse created successfully',
                    'data'    => $response->json(),
                ]);
            } else {
                return response()->json([
                    'status'  => false,
                    'message' => 'Delhivery API error',
                    'error'   => $response->json(),
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json([
                'status'  => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function updateDelhiveryWarehouse(Request $request)
    {
        $validated = $request->validate([
            'phone'           => 'nullable|string|max:15',
            'name'            => 'required|string',
            'pin'             => 'required|digits_between:4,10',
            'address'         => 'nullable',
        ]);

        try {
            $baseUrl = config('services.delhivery.base_url');
            $warehouseUrl = config('services.delhivery.update_warehouse_url');
            $url = "{$baseUrl}{$warehouseUrl}";
            $token = config('services.delhivery.token');

            $response = Http::withHeaders([
                'Authorization' => 'Token ' . $token,
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ])->post($url, $validated);

            if ($response->successful()) {
                return response()->json([
                    'status'  => true,
                    'message' => 'Warehouse updated successfully',
                    'data'    => $response->json(),
                ]);
            } else {
                return response()->json([
                    'status'  => false,
                    'message' => 'Delhivery API error',
                    'error'   => $response->json(),
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json([
                'status'  => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}
