<?php

namespace App\Services\Logistics\Providers;

use App\Adapters\Shipping\FedexAdapter;
use App\Models\Pickup;
use App\Models\Shipment;
use App\Services\Logistics\LogisticsProviderInterface;
use App\Traits\LogisticsHttpClientTrait;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FedExService implements LogisticsProviderInterface
{
    /**
     * Calculate shipping cost.
     *
     * @param array $packageDetails
     * @return array
     */
    use LogisticsHttpClientTrait;

    public function calculateShippingCost(array $packageDetails): array
    {
        $formattedDetails = FedExAdapter::formatShippingCostRequest($packageDetails);

        $baseUrl = rtrim(config('services.fedex.base_url'), '/');
        $path = ltrim(config('services.fedex.shipping_cost_url'), '/');
        $url = $baseUrl . '/' . $path;

        $headers = [
            'Authorization' => 'Bearer ' . $this->getFedexToken(),
            'Content-Type' => 'application/json',
        ];

        $body = json_encode($formattedDetails);

        // Log request body
        Log::info('Request Body:', $formattedDetails);

        $response = $this->sendPostRequest($url, $headers, $body);

        $responseArray = json_decode($response, true);

        // Log response array
        Log::info('Response Array:', $responseArray);

        $rates = [];

        if (isset($responseArray['output']['rateReplyDetails']) && is_array($responseArray['output']['rateReplyDetails'])) {
            $rates = [
                'logistics' => 'FedEx',
                // 'packagingTypes' => [
                //     'FEDEX_ENVELOPE',
                //     'FEDEX_PAK',
                //     'FEDEX_SMALL_BOX',
                //     'FEDEX_MEDIUM_BOX',
                //     'FEDEX_LARGE_BOX',
                //     'FEDEX_EXTRA_LARGE_BOX',
                //     'YOUR_PACKAGING'
                // ],
                // 'pickupTypes' => [
                //     'USE_SCHEDULED_PICKUP',
                //     'REGULAR_PICKUP',
                //     'ON_CALL_PICKUP',
                //     'DROPOFF_AT_FEDEX_LOCATION'
                // ],
                'data' => []
            ];

            foreach ($responseArray['output']['rateReplyDetails'] as $quote) {
                $rates['data'][] = [
                    'serviceType' => $quote['serviceType'] ?? 'Service type not available',
                    'serviceName' => $quote['serviceName'] ?? 'Service name not available',
                    'message' => $quote['customerMessages'][0]['message'] ?? 'Service message not available',
                    'amount' => $quote['ratedShipmentDetails'][0]['totalNetCharge'] ?? 'Amount not available',
                    'deliveryDays' => $quote['operationalDetail']['transitTime'] ?? 'Delivery days not available',
                    'provider' => "FedEx",
                ];
            }

            return $rates;
        }

        return ['error' => 'No rate quotes available from FedEx.'];
        // return ['error' => json_encode($response)];
    }

    /**
     * Track shipment.
     *
     * @param string $trackingNumber
     * @return array
     */
    public function trackShipment(array $trackingDetails): array
    {
        try {
            $fedexToken = $this->getFedExTrackingToken();

            $fedexFormattedData = FedExAdapter::formatTrackShipmentRequest($trackingDetails);

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.shipment_tracking_url'), $fedexFormattedData);

            $responseBody = $fedexResponse->json();

            if (
                $fedexResponse->successful() &&
                isset($responseBody['output']['completeTrackResults'][0]['trackResults'][0])
            ) {
                return [
                    'status'  => 'success',
                    'source'  => 'FedEx',
                    'message' => 'Tracking data retrieved successfully',
                    'data'    => FedExAdapter::simplifyTrackingResponse($responseBody)
                ];
            }

            return [
                'status'  => 'error',
                'source'  => 'FedEx',
                'message' => $responseBody['output']['alerts'][0]['message']
                    ?? 'Unable to retrieve tracking data',
                'data'    => $responseBody
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Shipment Tracking API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx Tracking API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    /**
     * Create shipment.
     *
     * @param array $shipmentDetails
     * @return array
     */
    public function createShipment(array $shipmentDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $fedexFormattedData = FedExAdapter::formatCreateShipmentRequest($shipmentDetails);

            $shipment = new Shipment();
            $shipment->provider = "Fedex";
            $shipment->request_body = json_encode($shipmentDetails);
            $shipment->formatted_request = json_encode($fedexFormattedData);

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.create_shipment_url'), $fedexFormattedData);

            $responseData = $fedexResponse->json();

            // return $responseData;
            $shipment->api_response = json_encode($fedexResponse->json());

            if ($fedexResponse->successful() && isset($fedexResponse['output']['transactionShipments'][0])) {
                $shipment_response = $fedexResponse['output']['transactionShipments'][0];

                $trackingNumbers = collect($shipment_response['pieceResponses'] ?? [])
                    ->pluck('trackingNumber')
                    ->filter()
                    ->values()
                    ->all();

                $shipment->tracking_no = json_encode($trackingNumbers);

                // $shipment->tracking_no = json_encode($shipment_response['pieceResponses'][0]['trackingNumber']) ?? null;
                $shipment->save();

                return [
                    'status' => 'success',
                    'source' => 'FedEx',
                    'message' => 'Shipment created successfully',
                    'labelUrl' => $shipment_response['pieceResponses'][0]['packageDocuments'][0]['url'] ?? null,
                    'trackingNumber' => $shipment_response['pieceResponses'][0]['trackingNumber'] ?? null,
                    'shipmentDetails' => FedExAdapter::simplifyShipmentResponse($shipment_response),
                ];
            }

            $shipment->save();

            if (isset($responseData['errors'])) {
                return FedExAdapter::formatErrorResponse($responseData);
            }
            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to create shipment',
                'data' => $fedexResponse->json()
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Shipment API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    public function createPickup(array $pickupDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $fedexFormattedData = FedExAdapter::formatPickupRequest($pickupDetails);

            $pickup = new Pickup();
            $pickup->provider = "Fedex";
            $pickup->request_body = json_encode($pickupDetails);
            $pickup->formatted_request = json_encode($fedexFormattedData);

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.create_pickup_url'), $fedexFormattedData);

            $responseData = $fedexResponse->json();

            // return $responseData;
            $pickup->api_response = json_encode($fedexResponse->json());

            if ($fedexResponse->successful() && isset($responseData['output']['pickupConfirmationCode'])) {
                $pickup->pickup_confirmation_code = $responseData['output']['pickupConfirmationCode'];
                $pickup->pickup_location_code = $responseData['output']['location'] ?? null;
                $pickup->save();

                return [
                    'status' => 'success',
                    'source' => 'FedEx',
                    'message' => 'Pickup scheduled successfully',
                    'pickupConfirmationCode' => $responseData['output']['pickupConfirmationCode'],
                    'location' => $responseData['output']['location'] ?? null,
                ];
            }

            $pickup->save();

            if (isset($responseData['errors'])) {
                return FedExAdapter::formatErrorResponse($responseData);
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to schedule pickup',
                'data' => $responseData,
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Pickup API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null,
            ];
        }
    }

    public function cancelPickup(array $pickupDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $fedexFormattedData = FedExAdapter::formatCancelPickupRequest($pickupDetails);

            $pickup = Pickup::where(['provider' => "Fedex", 'is_canceled' => 0, 'pickup_confirmation_code' => $pickupDetails['pickupConfirmationCode'], 'pickup_location_code' => $pickupDetails['location']])->first();

            $pickup->cancel_request = json_encode($fedexFormattedData);

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.cancel_pickup_url'), $fedexFormattedData);

            $responseData = $fedexResponse->json();

            // return $responseData;
            $pickup->cancel_response = json_encode($fedexResponse->json());

            if ($fedexResponse->successful() && isset($responseData['output']['pickupConfirmationCode'])) {
                $pickup->is_canceled = 1;
                $pickup->save();

                return [
                    'status' => 'success',
                    'source' => 'FedEx',
                    'message' => $responseData['output']['cancelConfirmationMessage']
                        ?? 'Pickup cancelled successfully',
                    'pickupConfirmationCode' => $responseData['output']['pickupConfirmationCode'],
                ];
            }

            $pickup->save();

            if (isset($responseData['errors'])) {
                return FedExAdapter::formatErrorResponse($responseData);
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to cancel pickup',
                'data' => $responseData,
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Pickup Cancellation API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null,
            ];
        }
    }

    public function returnShipment(array $shipmentDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $fedexFormattedData = FedExAdapter::formatReturnShipmentRequest($shipmentDetails);

            $shipment = new Shipment();
            $shipment->is_returned = 1;
            $shipment->provider = "Fedex";
            $shipment->request_body = json_encode($shipmentDetails);
            $shipment->formatted_request = json_encode($fedexFormattedData);

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.create_shipment_url'), $fedexFormattedData);

            $responseData = $fedexResponse->json();
            // dd($responseData);
            // return $responseData;
            $shipment->api_response = json_encode($fedexResponse->json());

            if ($fedexResponse->successful() && isset($fedexResponse['output']['transactionShipments'][0])) {
                $shipment_response = $fedexResponse['output']['transactionShipments'][0];

                $shipment->tracking_no = $shipment_response['pieceResponses'][0]['trackingNumber'] ?? null;
                $shipment->save();

                return [
                    'status' => 'success',
                    'source' => 'FedEx',
                    'message' => 'Shipment created successfully',
                    'labelUrl' => $shipment_response['pieceResponses'][0]['packageDocuments'][0]['url'] ?? null,
                    'trackingNumber' => $shipment_response['pieceResponses'][0]['trackingNumber'] ?? null,
                    'shipmentDetails' => FedExAdapter::simplifyShipmentResponse($shipment_response),
                ];
            }
            $shipment->save();

            if (isset($responseData['errors'])) {
                return FedExAdapter::formatErrorResponse($responseData);
            }
            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to create shipment',
                'data' => $fedexResponse->json()
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Return Shipment API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    public function cancelShipment(array $shipmentDetails): array
    {
        try {
            $fedexToken = $this->getFedexToken();

            $formattedRequest = FedExAdapter::formatCancelShipmentRequest($shipmentDetails);

            $shipment = Shipment::whereJsonContains('tracking_no' , $shipmentDetails['tracking_number'])->where('is_canceled', 0)->first();

            if ($shipment) {
                $fedexResponse = Http::withToken($fedexToken)
                    ->put(
                        config('services.fedex.base_url') . config('services.fedex.cancel_shipment_url'),
                        $formattedRequest
                    );

                $responseBody = $fedexResponse->json();


                $shipment->cancel_request = json_encode($formattedRequest);
                $shipment->cancel_response = json_encode($responseBody);
                if (
                    $fedexResponse->successful() &&
                    isset($responseBody['output']['cancelledShipment']) &&
                    $responseBody['output']['cancelledShipment'] === true
                ) {
                    $shipment->is_canceled = 1;
                }
                $shipment->save();

                if (
                    $fedexResponse->successful() &&
                    isset($responseBody['output']['cancelledShipment']) &&
                    $responseBody['output']['cancelledShipment'] === true
                ) {
                    return [
                        'status' => 'success',
                        'source' => 'FedEx',
                        'message' => $responseBody['output']['message'] ?? 'Shipment cancelled successfully',
                        'data' => [
                            'cancelledShipment' => true,
                            'cancelledHistory' => $responseBody['output']['cancelledHistory'] ?? null,
                            'transactionId' => $responseBody['transactionId'] ?? null,
                        ],
                    ];
                }
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => $responseBody['output']['message']
                    ?? 'Shipment cancellation failed (possibly already cancelled or invalid tracking number)',
                'data' => [
                    'cancelledShipment' => $responseBody['output']['cancelledShipment'] ?? false,
                    'cancelledHistory' => $responseBody['output']['cancelledHistory'] ?? false,
                    'transactionId' => $responseBody['transactionId'] ?? null,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Shipment Cancellation API exception', ['message' => $e->getMessage()]);

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    public function availablityCheck(array $options): array
    {
        try {
            $fedexRequest = FedexAdapter::buildFedexAddress($options);
            $fedexToken = $this->getFedexToken();

            $fedexResponse = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.address_validation_url'), $fedexRequest);

            if ($fedexResponse->successful() && !empty($fedexResponse['output']['resolvedAddresses'][0])) {

                $resolved = $fedexResponse['output']['resolvedAddresses'][0];
                $classification = $resolved['classification'] ?? '';
                $customerMessages = $resolved['customerMessages'] ?? [];
                $countrySupported = $resolved['attributes']['CountrySupported'] ?? 'false';

                $isSuccess = (
                    strtoupper($classification) === 'UNKNOWN' &&
                    empty($customerMessages) &&
                    $countrySupported === 'true'
                );

                if ($isSuccess) {
                    return [
                        'status' => 'success',
                        'source' => 'FedEx',
                        'address' => [
                            'streetLines' => $resolved['streetLinesToken'] ?? [],
                            'city'        => $resolved['cityToken'][0]['value'] ?? '',
                            'state'       => $resolved['stateOrProvinceCode'] ?? '',
                            'postalCode'  => $this->buildPostalCode(
                                $resolved['parsedPostalCode']['base'] ?? '',
                                $resolved['parsedPostalCode']['addOn'] ?? ''
                            ),
                            'countryCode' => $resolved['countryCode'] ?? '',
                        ],
                        'type' => $resolved['classification'] ?? 'UNKNOWN',
                    ];
                }


                $errors = [];

                foreach ($customerMessages as $msg) {
                    $errors[] = [
                        'code' => $msg['code'] ?? 'UNKNOWN',
                        'message' => $msg['message'] ?? 'Unknown error'
                    ];
                }

                return [
                    'status' => 'error',
                    'source' => 'FedEx',
                    'message' => 'Failed to resolve address via FedEx',
                    'data' => [
                        'response' => [
                            'errors' => $errors
                        ]
                    ]
                ];
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to resolve address via FedEx',
                'data' => $fedexResponse->json() // optional: include raw data
            ];
        } catch (\Exception $e) {
            Log::error('FedEx API exception', ['message' => $e->getMessage()]);
            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    public function checkPickupAvailability(array $options): array
    {
        try {
            $fedexRequest = FedexAdapter::buildPickupAvailabilityRequest($options);

            $fedexToken = $this->getFedexToken();

            Log::debug('FedEx Pickup Request', ['request' => $fedexRequest]);

            $response = Http::withToken($fedexToken)
                ->post(config('services.fedex.base_url') . config('services.fedex.pickup_availability_url'), $fedexRequest);

            if ($response->successful()) {
                $responseData = $response->json();

                $availableSlots = collect($responseData['output']['options'] ?? [])
                    ->filter(fn($item) => is_array($item) && ($item['available'] ?? false) === true)
                    ->map(fn($item) => [
                        'pickupDate'         => $item['pickupDate'] ?? null,
                        'cutoffTime'         => $item['cutOffTime'] ?? null,
                        'scheduleDay'        => $item['scheduleDay'] ?? null,
                        'defaultReadyTime'   => $item['defaultReadyTime'] ?? null,
                        'defaultLatestTime'  => $item['defaultLatestTimeOptions'] ?? null,
                    ])
                    ->values();

                return [
                    'status' => $availableSlots->isNotEmpty() ? 'success' : 'error',
                    'source' => 'FedEx',
                    'message' => $availableSlots->isNotEmpty()
                        ? 'Pickup availability found'
                        : 'No pickup availability',
                    'availableSlots' => $availableSlots,
                    'raw' => $responseData,
                ];
            }

            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'Failed to check pickup availability via FedEx',
                'data' => $response->json()
            ];
        } catch (\Exception $e) {
            Log::error('FedEx Pickup Availability Exception', ['message' => $e->getMessage()]);
            return [
                'status' => 'error',
                'source' => 'FedEx',
                'message' => 'FedEx API Exception: ' . $e->getMessage(),
                'data' => null
            ];
        }
    }

    private function getFedexToken()
    {
        $response = Http::asForm()->post(config('services.fedex.base_url') . config('services.fedex.token_url'), [
            'grant_type' => 'client_credentials',
            'client_id' => config('services.fedex.client_id'),
            'client_secret' => config('services.fedex.client_secret'),
        ]);

        if ($response->failed()) {
            abort(500, 'FedEx token generation failed');
        }

        return $response->json()['access_token'];
    }

    protected function getFedExTrackingToken(): string
    {
        $response = Http::asForm()->post(config('services.fedex.base_url') . config('services.fedex.token_url'), [
            'grant_type' => 'client_credentials',
            'client_id' => config('services.fedex.tracking_client_id'),
            'client_secret' => config('services.fedex.tracking_client_secret'),
        ]);

        if ($response->failed()) {
            abort(500, 'FedEx token generation failed');
        }

        return $response->json()['access_token'];
    }

    private function buildPostalCode($base, $addon)
    {
        return $addon ? "$base-$addon" : $base;
    }
}
