<?php

namespace App\Adapters\Shipping;

use Carbon\Carbon;

class FedexAdapter
{
    public static function buildFedexAddress(array $input): array
    {
        return [
            "inEffectAsOfTimestamp" => Carbon::now()->toDateString(),
            "validateAddressControlParameters" => [
                "includeResolutionTokens" => true
            ],
            "addressesToValidate" => [
                [
                    "address" => [
                        "streetLines" => $input['street'],
                        "city" => $input['city'],
                        "stateOrProvinceCode" => $input['state'],
                        "postalCode" => $input['postal_code'],
                        "countryCode" => $input['country'],
                    ],
                    "clientReferenceId" => "None"
                ]
            ]
        ];
    }

    public static function buildPickupAvailabilityRequest(array $input): array
    {
        return [
            'pickupAddress' => [
                'streetLines' => array_slice($input['pickupAddress']['streetLines'] ?? [], 0, 2),
                'city' => $input['pickupAddress']['city'] ?? '',
                'stateOrProvinceCode' => $input['pickupAddress']['stateOrProvinceCode'] ?? '',
                'postalCode' => $input['pickupAddress']['postalCode'] ?? '',
                'countryCode' => $input['pickupAddress']['countryCode'] ?? '',
            ],
            'carrierCode' => strtoupper($input['carrierCode'] ?? 'FDXE'),
            'pickupRequestType' => $input['pickupRequestType'] ?? ['SAME_DAY'],
            'localityType' => strtoupper($input['localityType'] ?? 'DOMESTIC'),
            'dispatchDate' => $input['dispatchDate'] ?? now()->format('Y-m-d'),
            'packageReadyTime' => $input['packageReadyTime'] ?? '15:30:00',
            'customerCloseTime' => $input['customerCloseTime'] ?? '18:00:00',
            'pickupType' => $input['pickupType'] ?? 'ON_CALL',
            'carriers' => $input['carriers'] ?? ['FDXE'],
            'countryRelationship' => strtoupper($input['countryRelationship'] ?? 'DOMESTIC')
        ];
    }

    public static function formatCreateShipmentRequest(array $input): array
    {
        return [
            "accountNumber" => [
                "value" => config('services.fedex.account_number')
            ],
            "labelResponseOptions" => "URL_ONLY",
            "requestedShipment" => [
                "shipDateStamp" => $input['shipment']['date'],
                "pickupType" => $input['shipment']['pickup_type'],
                "serviceType" => $input['shipment']['service_type'],
                "packagingType" => $input['shipment']['packaging_type'],

                "shipper" => [
                    "contact" => [
                        "personName" => $input['shipper']['name'],
                        "phoneNumber" => $input['shipper']['phone'],
                    ],
                    "address" => [
                        "streetLines" => is_array($input['shipper']['address'])
                            ? $input['shipper']['address']   // already split into multiple lines
                            : [$input['shipper']['address']], // wrap single line into an array
                        // "streetLines" => [$input['shipper']['address']],
                        "city" => $input['shipper']['city'],
                        "stateOrProvinceCode" => $input['shipper']['state'],
                        "postalCode" => $input['shipper']['postal_code'],
                        "countryCode" => $input['shipper']['country'],
                    ]
                ],

                "recipients" => [
                    [
                        "contact" => [
                            "personName" => $input['recipient']['name'],
                            "phoneNumber" => $input['recipient']['phone'],
                        ],
                        "address" => [
                            "streetLines" => is_array($input['recipient']['address'])
                                ? $input['recipient']['address']   // already split into multiple lines
                                : [$input['recipient']['address']], // wrap single line into an array
                            // "streetLines" => [$input['recipient']['address']],
                            "city" => $input['recipient']['city'],
                            "stateOrProvinceCode" => $input['recipient']['state'],
                            "postalCode" => $input['recipient']['postal_code'],
                            "countryCode" => $input['recipient']['country'],
                            "residential" => false
                        ]
                    ]
                ],

                "shippingChargesPayment" => [
                    "paymentType" => "SENDER",
                    "payor" => [
                        "accountNumber" => [
                            "value" => config('services.fedex.account_number')
                        ]
                    ]
                ],

                "customsClearanceDetail" => [
                    "commodities" => array_map(function ($custom, $index) use ($input) {
                        $package = $input['packages'][$index] ?? $input['packages'][0];
                        return [
                            "description" => $custom['description'],
                            "name" => explode(" ", $custom['description'])[0] ?? $custom['description'],
                            "countryOfManufacture" => $custom['country_of_manufacture'],
                            "quantity" => $custom['quantity'],
                            "quantityUnits" => "PCS",
                            "weight" => [
                                "units" => $package['weight_unit'],
                                "value" => $package['weight']
                            ],
                            "unitPrice" => [
                                "amount" => $custom['price_value'],
                                "currency" => $custom['currency']
                            ],
                            "customsValue" => [
                                "amount" => $custom['price_value'] * $custom['quantity'],
                                "currency" => $custom['currency']
                            ]
                        ];
                    }, $input['customs'], array_keys($input['customs'])),

                    "commercialInvoice" => [
                        "shipmentPurpose" => "SOLD"
                    ],
                    "dutiesPayment" => [
                        "paymentType" => "SENDER",
                        "payor" => [
                            "accountNumber" => [
                                "value" => config('services.fedex.account_number')
                            ]
                        ]
                    ]
                ],

                "labelSpecification" => [
                    "imageType" => "PDF",
                    "labelFormatType" => "COMMON2D"
                ],

                "requestedPackageLineItems" => array_map(function ($package, $index) {
                    return [
                        "sequenceNumber" => $index + 1,
                        "groupPackageCount" => 1,
                        "weight" => [
                            "units" => $package['weight_unit'],
                            "value" => $package['weight']
                        ],
                        "dimensions" => [
                            "length" => $package['dimensions']['length'],
                            "width" => $package['dimensions']['width'],
                            "height" => $package['dimensions']['height'],
                            "units" => $package['dimensions_unit']
                        ],
                        "customerReferences" => [
                            [
                                "value" => $package['customer_references']['value'] ?? "ORDER-PACK" . ($index + 1),
                                "type" => $package['customer_references']['type'] ?? "INVOICE"
                            ]
                        ]
                    ];
                }, $input['packages'], array_keys($input['packages']))
            ]
        ];
    }

    public static function formatPickupRequest(array $input): array
    {
        return [
            "associatedAccountNumber" => [
                "value" => config('services.fedex.account_number'),
            ],
            "associatedAccountNumberType" => "FEDEX_GROUND", // or FDXE based on account
            "originDetail" => [
                "pickupAddressType" => "ACCOUNT",
                "pickupLocation" => [
                    "contact" => [
                        "companyName" => $input['pickupLocation']['company'] ?? '',
                        "personName"  => $input['pickupLocation']['contactName'],
                        "phoneNumber" => $input['pickupLocation']['phone'],
                    ],
                    "address" => [
                        "streetLines"         => $input['pickupLocation']['address']['streetLines'],
                        "city"                => $input['pickupLocation']['address']['city'],
                        "stateOrProvinceCode" => $input['pickupLocation']['address']['state'],
                        "postalCode"          => $input['pickupLocation']['address']['postalCode'],
                        "countryCode"         => $input['pickupLocation']['address']['country'],
                    ],
                    "deliveryInstructions"    => $input['deliveryInstructions'] ?? null,
                ],
                "readyDateTimestamp" => $input['pickupDate'],
                "customerCloseTime"  => $input['closeTime'],
                "pickupDateType"     => "SAME_DAY",
                "packageLocation"    => $input['packageLocation'],
            ],
            "totalWeight" => [
                "units" => $input['totalWeight']['units'],
                "value" => $input['totalWeight']['value'],
            ],
            "packageCount" => $input['packageCount'],
            "carrierCode"  => $input['carrierCode'],
            "trackingNumber" => $input['trackingNumber'] ?? null,
            "remarks" => $input['remarks'] ?? null,
            "pickupChargesPayment" => [
                "paymentType" => "ACCOUNT",
                "payor" => [
                    "responsibleParty" => [
                        "accountNumber" => [
                            "value" => config('services.fedex.account_number'),
                        ],
                    ],
                ],
                "payorType" => "SENDER",
            ],
        ];
    }

    public static function formatCancelPickupRequest(array $input): array
    {
        return [
            "associatedAccountNumber" => [
                "value" => config('services.fedex.account_number'),
            ],
            "pickupConfirmationCode" => $input['pickupConfirmationCode'],
            "carrierCode"            => $input['carrierCode'],
            "scheduledDate"          => $input['scheduledDate'],
            "location"               => $input['location'],
            "remarks"                => $input['remarks'] ?? null,
        ];
    }

    public static function formatReturnShipmentRequest(array $input): array
    {
        $request = [
            "accountNumber" => [
                "value" => config('services.fedex.account_number')
            ],
            "labelResponseOptions" => "URL_ONLY",
            "requestedShipment" => [
                "shipDateStamp" => $input['shipment']['date'],
                "pickupType" => $input['shipment']['pickup_type'],
                "serviceType" => $input['shipment']['service_type'],
                "packagingType" => $input['shipment']['packaging_type'],

                "specialServicesRequested" => [
                    "specialServiceTypes" => ["RETURN_SHIPMENT"],
                    "returnShipmentDetail" => [
                        "returnType" => "PRINT_RETURN_LABEL"
                    ]
                ],

                "shipper" => [
                    "contact" => [
                        "personName" => $input['recipient']['name'],
                        "phoneNumber" => $input['recipient']['phone'],
                    ],
                    "address" => [
                        "streetLines" => is_array($input['recipient']['address'])
                            ? $input['recipient']['address']
                            : [$input['recipient']['address']],
                        // "streetLines" => [$input['recipient']['address']],
                        "city" => $input['recipient']['city'],
                        "stateOrProvinceCode" => $input['recipient']['state'],
                        "postalCode" => $input['recipient']['postal_code'],
                        "countryCode" => $input['recipient']['country'],
                    ]
                ],

                "recipients" => [
                    [
                        "contact" => [
                            "personName" => $input['shipper']['name'],
                            "phoneNumber" => $input['shipper']['phone'],
                        ],
                        "address" => [
                            "streetLines" => is_array($input['shipper']['address'])
                                ? $input['shipper']['address']
                                : [$input['shipper']['address']],
                            // "streetLines" => [$input['shipper']['address']],
                            "city" => $input['shipper']['city'],
                            "stateOrProvinceCode" => $input['shipper']['state'],
                            "postalCode" => $input['shipper']['postal_code'],
                            "countryCode" => $input['shipper']['country'],
                            "residential" => false
                        ]
                    ]
                ],

                "shippingChargesPayment" => [
                    "paymentType" => "SENDER",
                    "payor" => [
                        "accountNumber" => [
                            "value" => config('services.fedex.account_number')
                        ]
                    ]
                ],

                "customsClearanceDetail" => [
                    "commodities" => [
                        [
                            "description" => $input['customs']['description'],
                            "name" => $input['customs']['description'],
                            "quantity" => $input['customs']['quantity'],
                            "quantityUnits" => "PCS",
                            "countryOfManufacture" => $input['customs']['country_of_manufacture'],
                            "weight" => [
                                "units" => $input['package']['weight_unit'],
                                "value" => $input['package']['weight']
                            ],
                            "unitPrice" => [
                                "amount" => $input['customs']['value_usd'],
                                "currency" => $input['customs']['currency']
                            ],
                            "customsValue" => [
                                "amount" => $input['customs']['value_usd'],
                                "currency" => $input['customs']['currency']
                            ]
                        ]
                    ],
                    "dutiesPayment" => [
                        "paymentType" => "SENDER",
                        "payor" => [
                            "accountNumber" => [
                                "value" => config('services.fedex.account_number')
                            ]
                        ]
                    ]
                ],

                "labelSpecification" => [
                    "imageType" => "PDF",
                    "labelFormatType" => "COMMON2D"
                ],

                "requestedPackageLineItems" => array_map(function ($package, $index) {
                    return [
                        "sequenceNumber" => $index + 1,
                        "groupPackageCount" => 1,
                        "weight" => [
                            "units" => $package['weight_unit'],
                            "value" => $package['weight']
                        ],
                        "dimensions" => [
                            "length" => $package['dimensions']['length'],
                            "width" => $package['dimensions']['width'],
                            "height" => $package['dimensions']['height'],
                            "units" => $package['dimensions_unit']
                        ]
                    ];
                }, $input['packages'], array_keys($input['packages']))
            ]
        ];

        // Include referenceDetails if reference is provided
        if (!empty($input['reference'])) {
            $request['requestedShipment']['referenceDetails'] = [
                [
                    "referenceType" => "INVOICE_NUMBER",
                    "value" => $input['reference']
                ]
            ];
        }

        return $request;
    }

    public static function simplifyShipmentResponse(array $shipment): array
    {
        $pieces = $shipment['pieceResponses'] ?? [];
        $rate = $shipment['completedShipmentDetail']['shipmentRating']['shipmentRateDetails'][0] ?? [];
        $ops = $shipment['completedShipmentDetail']['operationalDetail'] ?? [];

        // Extract shipment level details
        $shipmentInfo = [
            'masterTrackingNumber' => $shipment['masterTrackingNumber'] ?? null,
            'serviceType'          => $shipment['serviceType'] ?? null,
            'shipDatestamp'        => $shipment['shipDatestamp'] ?? null,
            'serviceName'          => $shipment['serviceName'] ?? null,
            'shipmentDocuments'    => $shipment['shipmentDocuments'] ?? [],
        ];

        $simplifiedPieces = [];

        foreach ($pieces as $piece) {
            $customerReference = null;
            if (!empty($piece['customerReferences']) && isset($piece['customerReferences'][0]['value'])) {
                $customerReference = $piece['customerReferences'][0]['value'];
            }

            $simplifiedPieces[] = [
                'sequenceNumber'   => $piece['packageSequenceNumber'] ?? null,
                'trackingNumber'   => $piece['trackingNumber'] ?? null,
                'labelUrl'         => $piece['packageDocuments'][0]['url'] ?? null,
                'currency'         => $rate['currency'] ?? $piece['currency'] ?? 'USD',
                'totalBaseCharge'  => $rate['totalBaseCharge'] ?? null,
                'totalSurcharges'  => $rate['totalSurcharges'] ?? null,
                'totalNetCharge'   => $rate['totalNetCharge'] ?? null,
                'airport'          => $ops['airportId'] ?? null,
                'deliveryArea'     => $ops['destinationLocationId'] ?? null,
                'customerReference' => $customerReference,
            ];
        }

        return [
            'shipment' => $shipmentInfo,
            'packages' => $simplifiedPieces,
        ];
    }

    public static function simplifyShipmentResponse_old(array $shipment): array
    {
        $piece = $shipment['pieceResponses'][0] ?? [];
        $rate = $shipment['completedShipmentDetail']['shipmentRating']['shipmentRateDetails'][0] ?? [];
        $ops = $shipment['completedShipmentDetail']['operationalDetail'] ?? [];

        return [
            'trackingNumber'    => $piece['trackingNumber'] ?? null,
            'labelUrl'          => $piece['packageDocuments'][0]['url'] ?? null,
            'serviceType'       => $shipment['serviceType'] ?? null,
            'serviceName'       => $shipment['serviceName'] ?? null,
            'currency'          => $rate['currency'] ?? $piece['currency'] ?? 'USD',
            'totalBaseCharge'   => $rate['totalBaseCharge'] ?? null,
            'totalSurcharges'   => $rate['totalSurcharges'] ?? null,
            'totalNetCharge'    => $rate['totalNetCharge'] ?? null,
            'airport'           => $ops['airportId'] ?? null,
            'deliveryArea'      => $ops['destinationLocationId'] ?? null,
        ];
    }

    public static function simplifyTrackingResponse(array $tracking): array
    {
        $data = $tracking['output']['completeTrackResults'][0]['trackResults'][0] ?? [];

        return [
            'trackingNumber' => $data['trackingNumberInfo']['trackingNumber'] ?? null,
            'service'        => $data['serviceDetail']['description'] ?? null,
            'status'         => $data['latestStatusDetail']['description'] ?? null,
            'lastScanDate'   => $data['scanEvents'][0]['date'] ?? null,
            'lastScanStatus' => $data['scanEvents'][0]['eventDescription'] ?? null,
            'lastScanCity'   => $data['scanEvents'][0]['scanLocation']['city'] ?? null,
            'weightLB'       => $data['packageDetails']['weightAndDimensions']['weight'][0]['value'] ?? null,
            'weightKG'       => $data['packageDetails']['weightAndDimensions']['weight'][1]['value'] ?? null,
            'originCity'     => $data['originLocation']['locationContactAndAddress']['address']['city'] ?? null,
            'destinationCity' => $data['lastUpdatedDestinationAddress']['city'] ?? null,
            'shipperCity'    => $data['shipperInformation']['address']['city'] ?? null,
            'recipientCity'  => $data['recipientInformation']['address']['city'] ?? null
        ];
    }

    public static function formatCancelShipmentRequest(array $input): array
    {
        return [
            'accountNumber' => [
                'value' => config('services.fedex.account_number')
            ],
            'emailShipment' => 'false',
            'senderCountryCode' => $input['sender_country'],
            'deletionControl' => 'DELETE_ALL_PACKAGES',
            'trackingNumber' => $input['tracking_number']
        ];
    }

    public static function formatShippingCostRequest(array $shortRequest): array
    {
        return [
            'accountNumber' => [
                'value' => config('services.fedex.account_number')
            ],
            'requestedShipment' => [
                'shipper' => [
                    'address' => [
                        'postalCode' => $shortRequest['shipper_postal_code'],
                        'countryCode' => $shortRequest['shipper_country_code'],
                    ]
                ],
                'recipient' => [
                    'address' => [
                        'postalCode' => $shortRequest['recipient_postal_code'],
                        'countryCode' => $shortRequest['recipient_country_code'],
                    ]
                ],
                'pickupType' => $shortRequest['pickup_type'] === 'PICKUP'
                    ? 'REGULAR_PICKUP'
                    : 'DROPOFF_AT_FEDEX_LOCATION',
                'rateRequestType' => [
                    'ACCOUNT',
                    'LIST'
                ],
                'requestedPackageLineItems' => [
                    [
                        'weight' => [
                            'units' => $shortRequest['weight_unit'],
                            'value' => $shortRequest['weight_value'],
                        ]
                    ]
                ]
            ]
        ];
    }

    public static function formatTrackShipmentRequest(array $input): array
    {
        return [
            'includeDetailedScans' => (bool) $input['show_full_history'],
            'trackingInfo' => [
                [
                    'trackingNumberInfo' => [
                        'trackingNumber' => $input['tracking_number']
                    ]
                ]
            ]
        ];
    }

    public static function formatErrorResponse(array $response): array
    {
        $formattedErrors = [];

        if (!empty($response['errors'])) {
            foreach ($response['errors'] as $error) {
                $formattedErrors[] = [
                    'code' => $error['code'] ?? 'UNKNOWN',
                    'message' => $error['message'] ?? 'Unknown error',
                ];
            }
        }

        return [
            'status' => 'error',
            'source' => 'FedEx',
            'message' => 'Validation errors from FedEx',
            'errors' => $formattedErrors
        ];
    }

    public static function formatValidationErrorResponse(array $response): array
    {
        $formattedErrors = [];

        if (!empty($response['errors'])) {
            foreach ($response['errors'] as $field => $messages) {
                foreach ((array) $messages as $message) {
                    $formattedErrors[] = [
                        'code' => strtoupper(str_replace('.', '_', $field)),
                        'message' => $message,
                    ];
                }
            }
        }

        return [
            'status' => 'error',
            'source' => $response['provider'],
            'message' => 'Validation errors',
            'errors' => $formattedErrors,
            'error_type' => 'logistics_validation_error',
            'status_code' => 422,
        ];
    }
}
